<?php

class easyCommNotifier
{
    /** @var modX $modx */
    private $modx;

    private $mailConfig = [];

    /** @var modPHPMailer $mailService */
    private $mailService;

    function __construct(modX $modx)
    {
        $this->modx = $modx;

        $this->mailService = $this->modx->getService('mail', 'mail.modPHPMailer');

        $this->mailConfig[modMail::MAIL_SENDER] = $this->modx->getOption('ec_mail_from', null, $this->modx->getOption('emailsender'), true);
        $this->mailConfig[modMail::MAIL_FROM] = $this->modx->getOption('ec_mail_from', null, $this->modx->getOption('emailsender'), true);
        $this->mailConfig[modMail::MAIL_FROM_NAME] = $this->modx->getOption('ec_mail_from_name', null, $this->modx->getOption('site_name'), true);
    }

    /**
     * Уведомление при создании нового ecMessage или ecReply
     *
     * @param ecMessage|ecReply $object
     * @param bool $toManager Отправлять email менеджеру сайта?
     * @param bool $toUser Отправлять email автору?
     * @param bool $toParentUser Отправлять email об ответе автору "родительского" сообщения?
     * @return bool
     */
    public function notify($object, $toManager = false, $toUser = false, $toParentUser = false)
    {
        list($reply, $message, $thread, $resource) = $this->getRelatedObjects($object);
        if (!$message || !$thread || !$resource) {
            return false;
        }

        if($reply) {
            $propertiesKey = 'ecReplyForm';
            $data = $reply->toArray();
            $data['message'] = $message->toArray();
        } else {
            $propertiesKey = 'ecForm';
            $data = $message->toArray();
        }
        $data['thread'] = $thread->toArray();
        $data['resource'] = $resource->toArray();

        $data['site_manager_url'] = $this->getManagerSectionUrl();

        // В properties цепочки записаны все параметры, с которыми вызывался сниппет ecForm
        $snippetProperties = $this->getSnippetProperties($thread, $propertiesKey);

        if ($toManager && $this->modx->getOption('ec_mail_notify_manager', null, true)) {
            $this->sendEmailToManager($data, $snippetProperties);
        }
        if ($this->modx->getOption('ec_mail_notify_user', null, true)) {
            if ($toUser) {
                $this->sendEmailToUser($data, $snippetProperties);
            }
            if ($toParentUser && $reply) {
                /** @var xPDOObject $parentObject */
                $parentObject = $reply->getOne('ParentReply') ?: $reply->getOne('Message');

                $data = $parentObject->toArray();
                $data['reply'] = $reply->toArray();
                $data['thread'] = $thread->toArray();
                $data['resource'] = $resource->toArray();

                $this->sendEmailToUser($data, $snippetProperties);
            }
        }

        return true;
    }


    /**
     * @param array $data
     * @param array $snippetProperties
     *
     * @return void
     */
    private function sendEmailToUser($data, $snippetProperties)
    {
        $to = $data['user_email'];
        if (!easyCommValidator::isValidEmail($to)) {
            return;
        }
        $this->modx->log(modX::LOG_LEVEL_DEBUG, '[EasyComm] Send a email to the user ' . $to);

        $subjectTpl = $this->modx->getOption("emailSubjectUser", $snippetProperties, $this->modx->getOption("ec_mail_subject_user"), true);
        $bodyTpl = $this->modx->getOption("emailTplUser", $snippetProperties, '');

        $this->sendEmail(
            $to,
            $this->getEmailData($subjectTpl, $bodyTpl, $data)
        );
    }

    private function getEmailData($subjectTpl, $bodyTpl, $data) {
        /** @var pdoTools $pdoTools */
        $pdoTools = $this->modx->getService('pdoTools');

        return [
            modMail::MAIL_SUBJECT => $pdoTools->getChunk('@INLINE ' . $subjectTpl, $data),
            modMail::MAIL_BODY => $pdoTools->getChunk($bodyTpl, $data)
        ];
    }

    /**
     * @param array $data
     * @param array $snippetProperties
     *
     * @return void
     */
    private function sendEmailToManager($data, $snippetProperties)
    {
        // Почта менеджера указана в настройке ec_mail_manager
        $to = $this->modx->getOption('ec_mail_manager', null, '');
        // Параметр mailManager сниппета ecForm (ecReplyForm) переопределяет эту настройку
        $to = $this->modx->getOption('mailManager', $snippetProperties, $to, true);

        // Если везде пусто - письма пойдут на адрес, указанный в emailsender
        if (empty($to)) {
            $to = $this->modx->getOption('emailsender');
        }

        $this->modx->log(modX::LOG_LEVEL_DEBUG, '[EasyComm] Send a email to the manager ' . $to);

        $subjectTpl = $this->modx->getOption("emailSubjectManager", $snippetProperties, $this->modx->getOption("ec_mail_subject_manager"), true);
        $bodyTpl = $this->modx->getOption("emailTplManager", $snippetProperties, '');

        $this->sendEmail(
            $to,
            $this->getEmailData($subjectTpl, $bodyTpl, $data)
        );
    }


    /**
     * Send a email
     *
     * @param string $to
     * @param array $emailData
     *
     * @return void
     */
    private function sendEmail($to, $emailData)
    {
        if (empty($to)) {
            $this->modx->log(modX::LOG_LEVEL_WARN, '[EasyComm] Can`t send email, because recipient is empty');
        }
        $mailTo = array_map('trim', explode(',', $to));

        $this->mailService->reset();
        $this->mailService->setHTML(true);

        foreach ($emailData as $k => $v) {
            $this->mailService->set($k, $v);
        }

        foreach ($this->mailConfig as $key => $value) {
            $this->mailService->set($key, $value);
        }

        foreach ($mailTo as $mto) {
            $this->mailService->address('to', $mto);
        }

        if (!$this->mailService->send()) {
            $this->modx->log(modX::LOG_LEVEL_ERROR, '[EasyComm] An error occurred while trying to send the email: ' . $this->mailService->mailer->ErrorInfo);
        }
        $this->mailService->reset();
    }

    /**
     * @param $object
     * @return array
     */
    private function getRelatedObjects($object)
    {
        $reply = null;
        $message = null;
        $thread = null;
        $resource = null;
        if ($object instanceof ecReply) {
            $reply = $object;
            $message = $reply->getOne('Message');
        } elseif ($object instanceof ecMessage) {
            $message = $object;
        }
        if ($message) {
            /** @var ecThread $thread */
            $thread = $message->getOne('Thread');
            if ($thread) {
                /** @var modResource $resource */
                $resource = $thread->getOne('Resource');
            }
        }
        return [$reply, $message, $thread, $resource];
    }

    /**
     * @param ecThread $thread
     * @param string $propertiesKey
     * @return array
     */
    private function getSnippetProperties($thread, $propertiesKey = 'ecForm')
    {
        $properties = $thread->get('properties');
        return array_key_exists($propertiesKey, $properties)
            ? $properties[$propertiesKey]
            : [];
    }

    private function getManagerSectionUrl()
    {
        $manager_url = $this->modx->getOption('manager_url', null, MODX_MANAGER_URL);
        $site_url = $this->modx->getOption('site_url');
        return rtrim($site_url, '/') . '/' . ltrim($manager_url, '/');
    }
}
