<?php

require_once dirname(__FILE__) . '/easycommnotifier.class.php';
require_once dirname(__FILE__) . '/easycommvalidator.class.php';
require_once dirname(__FILE__) . '/easycommmediamanager.class.php';

/**
 * The base class for easyComm.
 */
class easyComm
{
    /* @var modX $modx */
    public $modx;

    /** @var array $config */
    public $config = [];

    /** @var array $initialized */
    private $initialized = array();

    /**
     * @param modX $modx
     * @param array $config
     */
    function __construct(modX $modx, array $config = array())
    {
        $this->modx = $modx;

        $corePath = $this->modx->getOption('ec_core_path', $config, $this->modx->getOption('core_path') . 'components/easycomm/');
        $assetsUrl = $this->modx->getOption('ec_assets_url', $config, $this->modx->getOption('assets_url') . 'components/easycomm/');
        $connectorUrl = $assetsUrl . 'connector.php';
        $actionUrl = $this->modx->getOption('ec_action_url', $config, $assetsUrl . 'action.php');

        $this->config = array_merge(array(
            'assetsUrl' => $assetsUrl,
            'cssUrl' => $assetsUrl . 'css/',
            'jsUrl' => $assetsUrl . 'js/',
            'imgUrl' => $assetsUrl . 'img/',
            'connectorUrl' => $connectorUrl,
            'actionUrl' => $actionUrl,

            'cultureKey' => $this->modx->getOption('cultureKey', $config, 'en'),
            'reCaptchaSiteKey' => $this->modx->getOption('ec_recaptcha2_site_key', $config, ''),
            'reCaptchaSecretKey' => $this->modx->getOption('ec_recaptcha2_secret_key', $config, ''),

            'corePath' => $corePath,
            'modelPath' => $corePath . 'model/',
            'chunksPath' => $corePath . 'elements/chunks/',
            'templatesPath' => $corePath . 'elements/templates/',
            'chunkSuffix' => '.chunk.tpl',
            'snippetsPath' => $corePath . 'elements/snippets/',
            'processorsPath' => $corePath . 'processors/',

            'json_response' => true,
            'nestedChunkPrefix' => 'ec_',
        ), $config);

        $this->modx->addPackage('easycomm', $this->config['modelPath']);
        $this->modx->lexicon->load('easycomm:default');
    }

    /**
     * @param array $config
     */
    public function setConfig($config)
    {
        $this->config = array_merge($this->config, $config);
    }

    /**
     * Initializes component into different contexts.
     *
     * @param string $ctx The context to load. Defaults to web.
     * @param array $scriptProperties
     *
     * @return boolean
     */
    public function initialize($ctx = 'web', $scriptProperties = array())
    {
        $this->config = array_merge($this->config, $scriptProperties);
        $this->config['ctx'] = $ctx;
        if (!empty($this->initialized[$ctx])) {
            return true;
        }
        switch ($ctx) {
            case 'mgr':
                break;
            default:
                if (!defined('MODX_API_MODE') || !MODX_API_MODE) {
                    /** @var pdoTools $pdoTools */
                    $pdoTools = $this->modx->getService('pdoTools');
                    $config = $pdoTools->makePlaceholders($this->config);
                    if ($css = $this->modx->getOption('ec_frontend_css')) {
                        $this->modx->regClientCSS(str_replace($config['pl'], $config['vl'], $css));
                    }
                    $config_js = array(
                        'ctx' => $ctx,
                        'jsUrl' => $this->config['jsUrl'] . 'web/',
                        'cssUrl' => $this->config['cssUrl'] . 'web/',
                        'imgUrl' => $this->config['imgUrl'] . 'web/',
                        'actionUrl' => $this->config['actionUrl'],
                        'reCaptchaSiteKey' => $this->config['reCaptchaSiteKey']
                    );
                    $config_js = json_encode($config_js);
                    $this->modx->regClientStartupScript('<script type="text/javascript">easyCommConfig = ' . $config_js . '</script>', true);
                    if ($js = trim($this->modx->getOption('ec_frontend_js'))) {
                        if (!empty($js) && preg_match('/\.js/i', $js)) {
                            $this->modx->regClientScript(str_replace($config['pl'], $config['vl'], $js));
                        }
                    }

                    if ($this->modx->getOption('ec_captcha_enable')) {
                        $reCaptcha2Api = trim($this->modx->getOption('ec_recaptcha2_api'));
                        if (!empty($reCaptcha2Api)) {
                            $this->modx->regClientHTMLBlock('<script src="' . str_replace($config['pl'], $config['vl'], $reCaptcha2Api) . '" async defer></script>');
                        }
                    }

                }
                $this->initialized[$ctx] = true;
                break;
        }
        return true;
    }


    public function getMediaManager($context = 'web')
    {
        $sourceId = (int)$this->modx->getOption('ec_files_source', null, 1);
        $mediaManager = new easyCommMediaManager($this->modx);
        $mediaManager->initialize($sourceId, $context);
        return $mediaManager;
    }

    /**
     * Create ecMessage through processor
     *
     * @param array $data $_POST
     *
     * @return array
     */
    public function createMessage($data = array())
    {
        // simple spam check
        if (!empty($this->config['antispamField'])) {
            if (!empty($data[$this->config['antispamField']])) {
                return $this->error("ec_fe_spam_detected");
            }
        }

        $requiredFields = array_map('trim', explode(',', $this->config['requiredFields']));
        $requiredFields = array_unique(array_merge($requiredFields, array('thread')));

        $allowedFields = array_map('trim', explode(',', $this->config['allowedFields']));
        $allowedFields = array_unique(array_merge($allowedFields, $requiredFields));

        $fields = array();
        foreach ($allowedFields as $field) {
            if (array_key_exists($field, $data)) {
                $fields[$field] = $this->sanitizeString($data[$field]);
            }
        }

        if (empty($fields['thread'])) {
            return $this->error('ec_message_err_save');
        }
        /** @var ecThread $thread */
        $thread = $this->modx->getObject('ecThread', array('name' => $fields['thread']));
        if (!$thread) {
            return $this->error('ec_message_err_save');
        }
        $fields['thread_id'] = $thread->get('id');

        $fields['requiredFields'] = $requiredFields;
        $fields['validateEmail'] = $this->config['validateEmail'];
        $fields['published'] = $this->getPublishedState();

        $response = $this->runProcessor('web/message/create', $fields);

        /* @var modProcessorResponse $response */
        if ($response->isError()) {
            return $this->error($response->getMessage(), $response->getFieldErrors());
        }

        /** @var ecMessage $message */
        $message = $this->modx->getObject('ecMessage', $response->response['object']['id']);
        if (!$message) {
            $this->modx->log(modX::LOG_LEVEL_ERROR,
                '[easyComm] Failed to get ecMessage object after creation');
            return $this->error('ec_message_err_save');
        }
        $thread = $message->getOne('Thread');
        $resource = $thread->getOne('Resource');

        if (!empty($this->config['files'])) {
            $allowedFileTypes = $this->getAllowedFileTypes($this->config['fileTypes']);
            $fileSize = intval($this->modx->getOption('fileSize', $this->config, 0));
            $filesCount = intval($this->modx->getOption('filesCount', $this->config, 0));

            $files = $this->getRequestFiles('files');

            $mediaManager = $this->getMediaManager($resource->get('context_key'));
            $path = $mediaManager->createContainer($resource->get('id'));
            if ($path !== false) {
                if (is_array($files)) {
                    foreach ($files as $file) {
                        // Прекратим загрузку файлов, если их кол-во больше максимально разрешенного
                        if ($filesCount > 0 && $message->get('files_count') >= $filesCount) {
                            $this->modx->log(modX::LOG_LEVEL_WARN, "[easyComm] The file '{$file['name']}' could not be loaded because max number of files is {$filesCount}");
                            break;
                        }
                        if($this->fileUploadAllowed($file, $allowedFileTypes, $fileSize)) {
                            $this->uploadFile($message, $file, $mediaManager, $path);
                        }
                    }
                }
            } else {
                $this->modx->log(modX::LOG_LEVEL_ERROR,
                    '[easyComm] Failed to create container for uploading files to the media source.');
            }
        }

        $notifier = new easyCommNotifier($this->modx);
        $notifier->notify($message, true, false, false);

        // Для вывода чанка нужно получить информацию о теме для корректного отображения звездочек
        $threadProperties = $thread->get('properties');
        $ecFormProperties = $threadProperties['ecForm'];
        $starsTheme = isset($ecFormProperties['starsTheme']) ? $ecFormProperties['starsTheme'] : 'default';
        $successData = array_merge($message->toArray(), ['stars_theme' => $starsTheme]);

        $successMessage = $message->get('published') ? 'ec_fe_send_success_published' : 'ec_fe_send_success_unpublished';

        if (!empty($this->config['tplSuccess'])) {
            /** @var pdoTools $pdoTools */
            $pdoTools = $this->modx->getService('pdoTools');
            return $this->success($successMessage, $pdoTools->getChunk($this->config['tplSuccess'], $successData));
        }
        return $this->success($successMessage, $successData);
    }

    /**
     * @param ecMessage $message
     * @return bool
     */
    public function mgrUploadFile(ecMessage &$message)
    {
        $resource = $message->getResource();
        if (!$resource) {
            $this->modx->log(modX::LOG_LEVEL_ERROR, '[easyComm] Message resource not found.');
            return false;
        }
        $mediaManager = $this->getMediaManager($resource->get('context_key'));
        $path = $mediaManager->createContainer($resource->get('id'));
        if ($path === false) {
            $this->modx->log(modX::LOG_LEVEL_ERROR, '[easyComm] Failed to create container.');
            return false;
        }

        // При загрузке файлов в админке приходит один файл, а не массив
        $file = $this->getRequestFiles('file');
        if ($file === false) {
            return false;
        }

        // Для менеджера мы НЕ выполняем проверку fileUploadAllowed(), пусть грузит что хочет.
        return $this->uploadFile($message, $file, $mediaManager, $path);
    }

    /**
     * Формирует массив с разрешенными типами файлов из строки.
     * Поддерживает сокращения:
     * "images" => [jpg, jpeg, png, gif, webp].
     * "documents" => [txt, doc, docx, xls, xlsx, pdf]
     *
     * @param $extensions string
     * @return array
     */
    private function getAllowedFileTypes($extensions)
    {
        $shorts = array(
            'images' => ['jpg', 'jpeg', 'png', 'gif', 'webp'],
            'documents' => array('txt', 'doc', 'docx', 'xls', 'xlsx', 'pdf')
        );
        $list = is_array($extensions) ? $extensions : explode(",", $extensions);
        $list = array_map('trim', $list);
        $result = array();
        foreach ($list as $ext) {
            if (in_array($ext, $shorts)) {
                $result = array_merge($result, $shorts[$ext]);
            } else {
                $result[] = $ext;
            }
        }
        return $result;
    }

    public function checkFiles($data = array())
    {
        if (empty($data['thread'])) {
            return $this->error('Bad request');
        }
        if (empty($data['files'])) {
            return $this->success('');
        }
        $thread = $this->modx->getObject('ecThread', array('name' => $data['thread']));
        if (!$thread) {
            return $this->error('Error! Thread not found!');
        }

        $properties = $thread->get('properties');
        $properties = array_key_exists('ecForm', $properties) ? $properties['ecForm'] : [];

        $allowedFileTypes = $this->getAllowedFileTypes($this->modx->getOption('fileTypes', $properties, '', true));
        if (!empty($allowedFileTypes)) {
            $fileSize = intval($this->modx->getOption('fileSize', $properties, 0));
            $filesCount = intval($this->modx->getOption('filesCount', $properties, 0));
            //$files = $this->modx->fromJSON($data['files']);
            $files = $data['files'];
            if (!is_array($files)) {
                return $this->error('Error! Bad data!');
            }
            $messages = array();
            $count = 0;
            foreach ($files as $file) {
                $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
                // TODO: локализация
                if ($fileSize > 0 && $file['size'] > $fileSize) {
                    $messages[] = 'Файл &laquo;' . $file['name'] . '&raquo; не будет загружен, он больше разрешенного размера.';
                    continue;
                }
                if ($filesCount > 0 && $count >= $filesCount) {
                    $messages[] = 'Файл &laquo;' . $file['name'] . '&raquo; не будет загружен, превышено ограничение на максимальное количество файлов.';
                    continue;
                }
                if (!in_array($ext, $allowedFileTypes)) {
                    $messages[] = 'Файл &laquo;' . $file['name'] . '&raquo; не будет загружен, такой тип файлов запрещен.';
                    continue;
                }
                $count++;
            }
            return $this->success('', $messages);
        }

        return $this->error('Error! Files is not allowed!');
    }

    /**
     * @param array $file
     * @param array $allowedTypes
     * @param int $maxSize
     * @return bool
     */
    private function fileUploadAllowed($file, $allowedTypes, $maxSize)
    {
        $fileName = $file['name'];
        $extension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));

        if (!in_array($extension, $allowedTypes)) {
            $this->modx->log(modX::LOG_LEVEL_WARN, "[easyComm] File with '$extension' extension is not allowed for user upload.");
            return false;
        }

        if ($maxSize > 0 && $file['size'] > $maxSize) {
            $this->modx->log(modX::LOG_LEVEL_WARN, "[easyComm] The file '$fileName' could not be loaded because it is too large (max=$maxSize bytes");
            return false;
        }

        return true;
    }

    /**
     * @param ecMessage $message
     * @param array $file
     * @param easyCommMediaManager $mediaManager
     * @param string $path
     * @return bool
     */
    private function uploadFile($message, $file, $mediaManager, $path)
    {
        $fileName = $file['name'];
        $extension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
        $internalName = $this->modx->filterPathSegment($this->generateRandomName()) . "." . $extension;

        $fileArray = $mediaManager->uploadObject(
            $path,
            array_merge($file, array('name' => $internalName, 'name_original' => $fileName))
        );
        if ($fileArray === false) {
            return false;
        }
        $message->appendFile($fileArray);
        $message->save();
        return true;
    }

    /**
     * @param $var
     * @return bool|array|mixed
     */
    private function getRequestFiles($var)
    {
        if (isset($_FILES[$var]) && isset($_FILES[$var]['name'])) {
            if (is_array($_FILES[$var]['name'])) {
                $result = array();
                $count = count($_FILES[$var]['name']);
                $keys = array_keys($_FILES[$var]);
                for ($i = 0; $i < $count; $i++) {
                    foreach ($keys as $key) {
                        if ($_FILES[$var]['error'][$i] == UPLOAD_ERR_OK) {
                            $result[$i][$key] = $_FILES[$var][$key][$i];
                        }
                    }
                }
                return $result;
            } else {
                if ($_FILES[$var]['error'] == UPLOAD_ERR_OK) {
                    return $_FILES[$var];
                }
            }
        }
        return false;
    }

    /**
     * Create ecReply through processor
     *
     * @param array $data $_POST
     *
     * @return array
     */
    public function createReply($data = array())
    {
        // simple spam check
        if (!empty($this->config['antispamField'])) {
            if (!empty($data[$this->config['antispamField']])) {
                return $this->error("ec_fe_spam_detected");
            }
        }

        $requiredFields = ['text', 'user_name', 'thread', 'message_id'];
        $allowedFields = ['text', 'user_name', 'user_email', 'thread', 'message_id', 'parent_id'];

        $fields = array();
        foreach ($allowedFields as $field) {
            if (array_key_exists($field, $data)) {
                $fields[$field] = $this->sanitizeString($data[$field]);
            }
        }

        if (empty($fields['thread'])) {
            return $this->error('ec_message_err_save');
        }
        /** @var ecThread $thread */
        $thread = $this->modx->getObject('ecThread', array('name' => $fields['thread']));
        if (!$thread) {
            return $this->error('ec_message_err_save');
        }
        $fields['thread_id'] = $thread->get('id');

        $fields['requiredFields'] = $requiredFields;
        $fields['validateEmail'] = $this->config['validateEmail'];
        $fields['published'] = $this->getPublishedState();

        $response = $this->runProcessor('web/reply/create', $fields);

        /** @var modProcessorResponse $response */
        if ($response->isError()) {
            return $this->error($response->getMessage(), $response->getFieldErrors());
        }

        /** @var ecReply $reply */
        $reply = $this->modx->getObject('ecReply', $response->response['object']['id']);
        if (!$reply) {
            $this->modx->log(modX::LOG_LEVEL_ERROR, '[EasyComm] Failed to get ecReply object after creation');
            return $this->error('ec_reply_err_save');
        }

        // Отправим уведомления:
        // 1. Менеджеру, что появился новый ответ
        // 2. Автору ответа - нет
        // 3. Автору сообщения, на которое ответ - да, если ответ автоматически опубликован
        $notifier = new easyCommNotifier($this->modx);
        $notifier->notify($reply, true, false, $reply->get('published'));
        if ($reply->get('published')) {
            $reply->set('notify_date', date('Y-m-d H:i:s'));
            $reply->save();
        }

        if (!empty($this->config['tplSuccess'])) {
            /** @var pdoTools $pdoTools */
            $pdoTools = $this->modx->getService('pdoTools');
            return $this->success('ec_fe_send_success', $pdoTools->getChunk($this->config['tplSuccess'], $reply->toArray()));
        }
        return $this->success('ec_fe_send_success', $reply->toArray());
    }

    /**
     * Получает статус публикации ecMessage или ecReply в зависимости от конфигурации
     */
    private function getPublishedState()
    {
        if (!empty($this->config['autoPublish'])) {
            switch ($this->config['autoPublish']) {
                case 'OnlyLogged':
                    if ($this->modx->user->hasSessionContext($this->modx->context->get('key'))) {
                        return true;
                    }
                    break;
                case 'All':
                    return true;
            }
        }
        return false;
    }

    /**
     * Генерирует случайное имя для файла
     * @param int $length
     * @return string
     */
    private function generateRandomName($length = 16)
    {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyz_';
        $charactersLength = strlen($characters);

        $result = '';

        for ($i = 0; $i < $length; $i++)
            $result .= $characters[rand(0, $charactersLength - 1)];

        return $result;
    }

    /**
     * @return array Массив с полями, которые являются рейтингом
     */
    public function getEcMessageRatingFields()
    {
        // список полей, которые являются рейтингом. Поле rating - всегда присутствует
        $ratingFields = $this->modx->getOption('ec_rating_fields', null, 'rating');
        $ratingFields = array_map('trim', explode(',', $ratingFields));
        return array_unique(array_merge(array('rating'), $ratingFields));
    }


    public function verifyCaptcha()
    {
        if (!$this->modx->getOption('ec_captcha_enable')) {
            return true;
        }

        return $this->verifyReCaptcha();
    }

    /**
     * Validate ReCaptcha
     *
     * @return boolean
     */
    private function verifyReCaptcha()
    {
        require_once($this->config['modelPath'] . 'recaptcha/autoload.php');
        $recaptcha = new \ReCaptcha\ReCaptcha($this->config['reCaptchaSecretKey'], new \ReCaptcha\RequestMethod\CurlPost());
        if (!($recaptcha instanceof \ReCaptcha\ReCaptcha)) {
            $this->modx->log(modX::LOG_LEVEL_ERROR, '[easyComm] Failed to load \ReCaptcha\ReCaptcha class.');
            return false;
        }

        $gResponse = $_POST['g-recaptcha-response'];
        $reCaptchaResponse = null;
        if (!empty($gResponse)) {
            $reCaptchaResponse = $recaptcha->verify($gResponse, $_SERVER["REMOTE_ADDR"]);
        }
        if ($reCaptchaResponse == null || !$reCaptchaResponse->isSuccess()) {
            return false;
        }

        return true;
    }


    public function voteMessage($data = array())
    {
        $messageId = isset($data['messageId']) ? (int)$data['messageId'] : 0;
        $value = isset($data['value']) ? (int)$data['value'] : 0;
        $propertiesKey = isset($data['propertiesKey']) ? $data['propertiesKey'] : false;

        if (empty($messageId) || empty($value) || empty($propertiesKey)) {
            return $this->error('ec_fe_vote_error_ns');
        }

        $properties = @$_SESSION['easyComm']['ecMessages'][$propertiesKey];
        if (empty($properties) || !is_array($properties) || empty($properties['votingEnable'])) {
            return $this->error('ec_fe_vote_error_ns');
        }

        $userId = $this->modx->user->get('id') ?: 0;
        if ((empty($properties['votingAllowGuest']) && empty($userId))) {
            return $this->error('ec_fe_vote_error_guest');
        }

        $response = $this->runProcessor('web/message/vote', array_merge($data, $properties));

        /* @var modProcessorResponse $response */
        if ($response->isError()) {
            return $this->error($response->getMessage(), $response->getFieldErrors());
        } else {
            /* @var ecMessage $message */
            if ($message = $this->modx->getObject('ecMessage', $messageId)) {
                $vote = $response->getObject();
                $value = 0;
                if (is_array($vote) && isset($vote['value'])) {
                    $value = $vote['value'];
                }

                return $this->success('ec_fe_vote_success', array(
                    'votes' => $message->get('votes'),
                    'likes' => $message->get('likes'),
                    'dislikes' => $message->get('dislikes'),
                    'votes_rating' => $message->get('votes_rating'),
                    'votes_rating_percent' => number_format($message->get('votes_rating') * 100, 3, '.', ''),
                    'messageId' => $messageId,
                    'value' => $value,
                ));
            }
        }

        return $this->error('ec_fe_vote_error_ns');
    }


    /**
     * Get the true client IP. Returns an array of values:
     *
     * * ip - The real, true client IP
     * * suspected - The suspected IP, if not alike to REMOTE_ADDR
     * * network - The client's network IP
     *
     * @access public
     * @return array
     */
    public function getClientIp()
    {
        $this->modx->getRequest();
        $ip = $this->modx->request->getClientIp();
        return $ip['ip'];
    }

    /**
     * Process and return the output from a Chunk by name.
     *
     * @param string $name The name of the chunk.
     * @param array $properties An associative array of properties to process the Chunk with, treated as placeholders within the scope of the Element.
     * @param boolean $fastMode If false, all MODX tags in chunk will be processed.
     *
     * @return string The processed output of the Chunk.
     */
    public function getChunk($name, array $properties = array(), $fastMode = false)
    {
        $pdoTools = $this->modx->getService('pdoTools');
        return $pdoTools->getChunk($name, $properties, $fastMode);
    }

    /**
     * Shorthand for the call of processor
     *
     * @access public
     * @param string $action Path to processor
     * @param array $data Data to be transmitted to the processor
     * @return mixed The result of the processor
     */
    public function runProcessor($action = '', $data = array())
    {
        if (empty($action)) {
            return false;
        }
        return $this->modx->runProcessor($action, $data, array('processors_path' => $this->config['processorsPath']));
    }

    /**
     * Sanitize MODX tags
     *
     * @param string $string Any string with MODX tags
     *
     * @return string String with html entities
     */
    public function sanitizeString($string = '')
    {
        $string = htmlentities(trim($string), ENT_QUOTES, "UTF-8");
        $string = preg_replace('/^@.*\b/', '', $string);
        $arr1 = array('[', ']', '`');
        $arr2 = array('&#091;', '&#093;', '&#096;');
        return str_replace($arr1, $arr2, $string);
    }


    /**
     * This method returns an error of the cart
     *
     * @param string $message A lexicon key for error message
     * @param array $data .Additional data, for example cart status
     * @param array $placeholders Array with placeholders for lexicon entry
     *
     * @return array|string $response
     */
    public function error($message = '', $data = array(), $placeholders = array())
    {
        $response = array(
            'success' => false,
            'message' => $this->modx->lexicon($message, $placeholders),
            'data' => $data
        );
        return $this->config['json_response']
            ? $this->modx->toJSON($response)
            : $response;
    }

    /**
     * This method returns an success of the action
     *
     * @param string $message A lexicon key for success message
     * @param array $data .Additional data, for example cart status
     * @param array $placeholders Array with placeholders for lexicon entry
     *
     * @return array|string $response
     * */
    public function success($message = '', $data = array(), $placeholders = array())
    {
        $response = array(
            'success' => true,
            'message' => $this->modx->lexicon($message, $placeholders),
            'data' => $data
        );
        return $this->config['json_response']
            ? $this->modx->toJSON($response)
            : $response;
    }

    /**
     * This method returns the list of thread fields.
     * @return array
     * */
    public function getThreadFields()
    {
        return array_keys($this->modx->getFields('ecThread'));
    }

    /**
     * This method returns the list of fields in the thread grid.
     * @return array
     * */
    public function getThreadGridFields()
    {
        $fields = array_keys($this->modx->getFields('ecThread'));

        $grid_fields = $this->modx->getOption('ec_thread_grid_fields');
        $grid_fields = array_map('trim', explode(',', $grid_fields));
        return array_values(array_intersect($grid_fields, $fields));
    }

    public function getThreadWindowFields()
    {
        $fields = array_keys($this->modx->getFields('ecThread'));

        $window_fields = $this->modx->getOption('ec_thread_window_fields');
        $window_fields = array_map('trim', explode(',', $window_fields));
        return array_values(array_intersect($window_fields, $fields));
    }

    /**
     * This method returns the list of message fields.
     * @return array
     * */
    public function getMessageFields()
    {
        $ecMessageFields = $this->modx->getFields('ecMessage');
        $otherFields = ['preview_url', 'thread_resource_id', 'thread_name', 'thread_title', 'resource_pagetitle', 'replies', 'replies_new'];
        return array_merge(array_keys($ecMessageFields), $otherFields);
    }

    /**
     * This method returns the list of fields in the message grid.
     * @return array
     * */
    public function getMessageGridFields()
    {
        $grid_fields = $this->modx->getOption('ec_message_grid_fields');
        $grid_fields = array_map('trim', explode(',', $grid_fields));
        return array_values(array_intersect($grid_fields, $this->getMessageFields()));
    }

    public function getMessageWindowLayout()
    {
        return $this->modx->getOption('ec_message_window_layout');
    }

    /**
     * Loads additional scripts for message form from easyComm plugins
     *
     * @return array
     * */
    public function getPluginsJS()
    {
        $result = array();
        if (!empty($this->modx->easyCommPlugins)) {
            foreach ($this->modx->easyCommPlugins->plugins as $plugin) {
                if (!empty($plugin['manager']['ecMessage'])) {
                    $result[] = $plugin['manager']['ecMessage'];
                }
                if (!empty($plugin['manager']['ecThread'])) {
                    $result[] = $plugin['manager']['ecThread'];
                }
            }
        }
        return $result;
    }

    public function humanFileSize($size, $precision = 2)
    {
        $units = array('B', 'kB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB');
        $step = 1024;
        $i = 0;
        while (($size / $step) > 0.9) {
            $size = $size / $step;
            $i++;
        }
        return round($size, $precision) . ' ' . $units[$i];
    }

    /**
     * Pathinfo function for cyrillic files
     *
     * @param $path
     * @param string $part
     *
     * @return array
     */
    public function cyrillicPathinfo($path, $part = '')
    {
        if (preg_match('#[а-яё]#im', $path)) {
            $path = strtr($path, ['\\' => '/']);

            preg_match('#[^/]+$#', $path, $file);
            preg_match('#([^/]+)[.$]+(.*)#', $path, $file_ext);
            preg_match('#(.*)[/$]+#', $path, $dirname);

            $info = [
                'dirname' => (isset($dirname[1]))
                    ? $dirname[1]
                    : '.',
                'basename' => $file[0],
                'extension' => (isset($file_ext[2]))
                    ? $file_ext[2]
                    : '',
                'filename' => (isset($file_ext[1]))
                    ? $file_ext[1]
                    : $file[0],
            ];
        } else {
            $info = pathinfo($path);
        }

        return !empty($part) && isset($info[$part])
            ? $info[$part]
            : $info;
    }

    /**
     * @param float $rating
     * @param int $ratingMax
     * @param float $starWidth
     * @param float $gapWidth
     * @return float|int
     */
    public function ratingToPercentage($rating, $ratingMax, $starWidth = 20, $gapWidth = 4)
    {
        // Ограничиваем рейтинг
        $rating = max(0, min($rating, $ratingMax));

        // если не заданы размеры звезд, то вернем простой процент
        if ($starWidth == 0 || $gapWidth == 0) {
            return $rating / $ratingMax * 100;
        }

        // целая часть
        $whole = floor($rating);
        $pixels = ($whole * $starWidth) + max(0, $whole - 1) * $gapWidth;

        // дробная часть
        $fraction = $rating - $whole;
        if ($fraction > 0) {
            $pixels += $gapWidth + ($fraction * $starWidth);
        }

        // Общая ширина
        $totalWidth = $starWidth * $ratingMax + $gapWidth * ($ratingMax - 1);

        return $pixels / $totalWidth * 100;
    }


}