<?php

class easyCommMediaManager
{
    /** @var modX $modx */
    private $modx;

    /** @var modMediaSource $mediaSource */
    private $mediaSource;

    /**
     * easyCommMediaManager constructor.
     * @param modX $modx
     */
    function __construct(modX $modx)
    {
        $this->modx = $modx;
    }

    /**
     * @param int $mediaSourceId
     * @param string $context
     * @return bool
     */
    function initialize($mediaSourceId, $context)
    {
        $this->modx->loadClass('sources.modMediaSource');
        $this->mediaSource = modMediaSource::getDefaultSource($this->modx, $mediaSourceId);
        if (!$this->mediaSource) {
            $this->modx->log(modX::LOG_LEVEL_FATAL, '[easyComm] Failed to get modMediaSource #' . $mediaSourceId);
            return false;
        }
        $this->mediaSource->set('ctx', $context);
        $this->mediaSource->initialize();
        return true;
    }

    /**
     * @param int $resource_id
     * @return string|bool
     */
    public function createContainer($resource_id)
    {
        if (!$this->mediaSource->checkPolicy('create')) {
            $this->modx->log(modX::LOG_LEVEL_ERROR, '[easyComm] No permission to upload files to a specified source.');
            return false;
        }

        // build path
        $internalPath = $this->getPathForSpecifiedResource($resource_id);
        $path = $this->mediaSource->getBasePath() . $internalPath;

        // Create directory if doesn't exists
        if ($internalPath !== "/" && $internalPath !== "") {
            if ($this->mediaSource->fileHandler) {
                $d = $this->mediaSource->fileHandler->make($path, array(), 'modDirectory');
                if (!$d->exists()) {
                    if (!$this->mediaSource->createContainer($internalPath, '')) {
                        $this->modx->log(modX::LOG_LEVEL_ERROR, '[easyComm] Can`t create directory: ' . $path);
                        return false;
                    }
                }
            } else {
                if (!$this->mediaSource->createContainer($internalPath, '')) {
                    $this->modx->log(modX::LOG_LEVEL_ERROR, '[easyComm] Can`t create directory: ' . $path);
                    return false;
                }
            }
        }

        return $internalPath;
    }

    /**
     * @param string $container
     * @param array $object
     * @return array|false
     */
    public function uploadObject($container, $object)
    {
        // TODO: Проверить код из ms2:
        // https://github.com/modx-pro/miniShop2/blob/master/core/components/minishop2/processors/mgr/gallery/upload.class.php

        if ($this->mediaSource->uploadObjectsToContainer($container, array($object))) {
            $thumbnailPath = '';
            $thumbnailUrl = '';
            if ($thumbnail = $this->createThumbnail($container, $object)) {
                $thumbnailPath = $container . $thumbnail;
                $thumbnailUrl = $this->mediaSource->getObjectUrl($thumbnailPath);
            }
            return array(
                'name' => $object['name'],
                'name_original' => $object['name_original'],
                'path' => $container . $object['name'],
                'url' => $this->mediaSource->getObjectUrl($container . $object['name']),
                'thumb_path' => $thumbnailPath,
                'thumb_url' => $thumbnailUrl,
                'size' => $object['size'],
                'extension' => pathinfo($object['name'], PATHINFO_EXTENSION)
            );
        } else {
            $errorMessage = '';
            $errors = $this->mediaSource->getErrors();
            foreach ($errors as $k => $msg) {
                $errorMessage .= $k . ': ' . $msg . '. ';
            }
            $this->modx->log(modX::LOG_LEVEL_ERROR, '[easyComm] Can`t upload object to media source: ' . $errorMessage);
            return false;
        }
    }

    /**
     * @param $objectPath
     * @return bool
     */
    public function removeObject($objectPath)
    {
        return $this->mediaSource->removeObject($objectPath);
    }

    /**
     * @param int $resource modResource id
     * @return array|string|string[]
     */
    private function getPathForSpecifiedResource($resource)
    {
        $path = $this->modx->getOption('ec_files_path', null, '/', true);

        $search = array('{year}', '{month}', '{day}', '{resource}');
        $replace = array(date('Y'), date('m'), date('d'), $resource);

        return str_replace($search, $replace, $path);
    }

    /**
     * @param $container
     * @param $object
     * @return string
     */
    private function createThumbnail($container, $object)
    {
        $filename = $container . $object['name'];

        $info = $this->mediaSource->getObjectContents($filename);
        if (!is_array($info)) {
            $this->modx->log(modX::LOG_LEVEL_ERROR,
                "[easyComm] Could not retrieve contents of file {$filename} from media source.");
            return '';
        }

        $properties = $this->mediaSource->getPropertyList();
        $imageExtensions = $this->modx->getOption('imageExtensions', $properties, 'jpg,jpeg,png,gif,webp');
        $imageExtensions = array_map('trim', explode(',', strtolower($imageExtensions)));

        $extension = pathinfo($object['name'], PATHINFO_EXTENSION);
        if (!in_array($extension, $imageExtensions)) {
            return '';
        }

        $thumbnailConfig = $this->modx->getOption('thumbnailConfig', $properties, 'h=80&w=80&far=1&zc=0&f=jpg');

        $phpThumb = $this->getThumbnailPhpThumb($thumbnailConfig);

        $tf = tempnam(MODX_BASE_PATH, 'ec_');
        file_put_contents($tf, $info['content']);
        $phpThumb->setSourceFilename($tf);

        $thumbnailFileName = $this->getThumbnailFileName($object['name']);
        try {
            if ($phpThumb->GenerateThumbnail() && $phpThumb->RenderOutput()) {
                $output = $phpThumb->outputImageData;

                if (!$this->mediaSource->createObject(
                    $container,
                    $thumbnailFileName,
                    $output
                )) {
                    $this->modx->log(modX::LOG_LEVEL_ERROR, '[easyComm] Could not save thumbnail: ' . $container . $thumbnailFileName);
                    return '';
                }
            }
        } catch (Exception $e) {
            $this->modx->log(modX::LOG_LEVEL_ERROR, '[easyComm] Could not generate thumbnail: ' . $e->getMessage());
        }

        if (file_exists($phpThumb->sourceFilename)) {
            @unlink($phpThumb->sourceFilename);
        }
        @unlink($tf);

        return $thumbnailFileName;
    }

    /**
     * @param string $thumbnailConfig
     */
    private function getThumbnailPhpThumb($thumbnailConfig)
    {
        if (!class_exists('modPhpThumb')) {
            if (file_exists(MODX_CORE_PATH . 'model/phpthumb/modphpthumb.class.php')) {
                /** @noinspection PhpIncludeInspection */
                require MODX_CORE_PATH . 'model/phpthumb/modphpthumb.class.php';
            } else {
                $this->modx->getService('phpthumb', 'modPhpThumb');
            }
        }

        $phpThumb = new modPhpThumb($this->modx);
        $phpThumb->initialize();

        $params = [];
        if (is_string($thumbnailConfig)) {
            parse_str($thumbnailConfig, $params);
        }
        foreach ($params as $k => $v) {
            $phpThumb->setParameter($k, $v);
        }
        return $phpThumb;
    }

    private function getThumbnailFileName($filename)
    {
        $pathinfo = pathinfo($filename);
        return $pathinfo['filename'] . '_thumb.' . $pathinfo['extension'];
    }

}
